/////////////////////////////////////////////////////////////////////////////////
//
// Shader  : TheEmu - Complex Functions.fsh
// Creator : TheEmu
// Version : 1.0.1
// Date    : 2015/06/08
// License : Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
//
// History:
//   1.0   - 2015/06/05 - Initial release.
//   1.0.1 - 2015/06/07 - Split large switch statement into smaller ones
//                        because the NVIDIA fragment shader compiler is
//                        limted to 64 cases in a switch statement.
//   1.0.2 - 2015/06/20 - Added a few more functions.
//
/////////////////////////////////////////////////////////////////////////////////
//
// This shader is based on the Complex Domains shader by Huw Bowles but has been
// greatly modified and extended.  It  generates images based on considering the
// window as a rectangular region of the complex plane and evaluating a function
// at each point.  The function results are used to either display the structure
// of the function in a mathematical way or as a distortion applied to a texture
// which tiles the window. The results may be optionally animated.
//
// The operation of the shader is controled by a number of parameters  that  may
// specified by uniform: clauses in a VGHD scene file.  These parameters are all
// described in comments below.
//
// Note, only the variables declared as uniform are inputs,  the other variables
// declared with them are simply constants derived from them.
//
/////////////////////////////////////////////////////////////////////////////////

// Standard shader inputs.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

/////////////////////////////////////////////////////////////////////////////////
//
// Possible Enhancements
//
//    Zoom into some particular location
//    Increased visibility for cuts
//    Non-standard cuts
//
/////////////////////////////////////////////////////////////////////////////////

// Control inputs. Part 1 - Common control parameters.

// FunctionId specifies which complex function is to be plotted.  See the set of
// FUNC_nn definitions below for the list of functions and their ids. FunctionId
// must always be specified.

uniform int FunctionId;

// By default only the function explicitly given by FunctionId is used. However,
// if a non-zero value for FunctionDuration is specified then  the  function  id
// will cycle through the available functions  changing  every  FunctionDuration
// seconds.

uniform float FunctionDuration;

// Optionally the function argument and result may both be modified by  applying
// constant scale factors and adding constants, i.e. the evaluation of fz = f(z)
// is replaced by fz" = (fz'+c)*d where fz' = f((z+a)*b) with a, b, c and d  all
// constants.  Note,  the * operators above do NOT denote complex multiplication
// only scalings of the x and y components of z by the components of the  second
// operand, i.e. component-wise multiplication.
// 
// It may be noted that the effect of these four modifiers on the final image is
//
//    PreScale  - zooms in or out of and/or stretches the image.
//    PreAdd    - causes the center of the image to be shifted.
//    PostScale - alters scale of the contours and colour bands.
//    PostAdd   - actualy affects the form of the function result.
//
// Note that of these modifiers only PostAdd causes any change to the  structure
// of the image, the others simply change scales or move its central point.

uniform vec2 PreAdd;    // The constant a, defaults to (0.0,0.0)
uniform vec2 PreScale;  // The constant b, defaults to (1.0,1.0)

uniform vec2 PostAdd;   // The constant c, defaults to (0.0,0.0)
uniform vec2 PostScale; // The constant d, defaults to (1.0,1.0)

   vec2 DomainScale   = ( PreScale  == vec2(0.0) ) ? vec2(1.0) : PreScale;
   vec2 CodomainScale = ( PostScale == vec2(0.0) ) ? vec2(1.0) : PostScale;

// By default an internal colouring scheme is used, however this may be replaced
// by an input texture by specifying a non zero value for ColourScheme which may
// take one of the following values
//
//    0 - Internal colourisation based on cartesian components (x,y)
//    1 - Internal colourisation based on polar components (r,theta)
//    2 - Internal colourisation based on polar components (r,theta)
//
//   10 - Texture based colourisation based on cartesian components (x,y)
//   11 - As 10 but with no tile flipping (see below)
//   12 - Texture based colourisation based on polar components (r,theta)
//   13 - As 12 but with no tile flipping (see below)
//
// where (x,y) are the real and imaginary parts of the complex  number z = x+i*y
// and (r,a) are the polar components of the same compex number z = r*exp(i*a).
//
// When using the texture based colouring schemes it is normal to flip alternate
// tiles so that there is at least continuity of colour where the tiles meet  as
// this produces a smoother image.  However, in some cases, this may be unwanted
// and so versions of these colouring schemes that do not perform this  flipping
// are also provided and these are useful when  the  textures  are  deliberately
// directional, e.g. have an arrow on them.

uniform int ColourScheme;

// By default the output is static, however this may be changed to a periodically
// varying colourisation by specifying its period in seconds.

uniform float ColourPeriod;

   float Alpha = (ColourPeriod==0.0) ? 0.0 : u_Elapsed * radians(360.0) / ColourPeriod;

   float SinAlpha = sin(Alpha);
   float CosAlpha = cos(Alpha);

   mat2 RotAlpha = mat2 ( CosAlpha, -SinAlpha, SinAlpha, CosAlpha );

/////////////////////////////////////////////////////////////////////////////////

// Control inputs. Part 2 - Parameters for the internal colour schemes.

// By default the contour lines are somewhat blurred for  artistic  effect.  The
// blurring can be changed by specifying a non zero value for  ContourSharpness.
// The first component of this parameter controls the sharpness of  the  contour
// drawn for the cartesian components of the function's  value  and  the  second
// controls the sharpness of the contour for the polar components. Higher values
// increase the sharpness with values of 10.0 produceing very fine lines and low
// values cause the contour lines to broaden and to increase in  fuzzyness.  The
// default value is 1.0 which is a compromise between artistry and precision.

uniform vec2 ContourSharpness;

   vec2 ContourSharp = 4.5 * ( (ContourSharpness==vec2(0.0)) ? vec2(1.0,1.0) : ContourSharpness );

// By default the contour each lines represent steps of 1.0 in the components of
// the function's value.  This can be changed by specifying a non-zero value for
// ContourDensity the components of which give the number of contour  lines  per
// unit step in the corresponding component of the function value.  In the order
// in  which they must be given in the  uniform: clause in the VGHD  scene  file
// the four components of ContourDensity control the x, y, r and  theta  contour
// spacing.  As  a special case if all four components of ContourDensity are 0.0
// is treated as if it were (1.0,1.0,1.0,0.0) so that the default  behaviour  is
// to plot the x, y and r contours but not those for theta.  This  is mainly for
// aesthetic reasons. If all four sets of contours are to be suppressed then use
// a negative value for at least one of ContourDensity's components. Normaly the
// components of ContourDensity should be small integers  (up to 10)  as  larger
// values result in the contour lines being too crowded and non  integer  values
// are inconvenient, especialy for the theta contours,  though larger values can
// be used for the theta contour density.

uniform vec4 ContourDensity;

   vec4 ContourXX = (ContourDensity==vec4(0.0)) ? vec4(1.0,1.0,1.0,0.0) : ContourDensity;

   vec4 ContourEnable = vec4 ( (ContourXX.x>0.0) ? 1.0 : 0.0,
                               (ContourXX.y>0.0) ? 1.0 : 0.0,
                               (ContourXX.z>0.0) ? 1.0 : 0.0,
                               (ContourXX.w>0.0) ? 1.0 : 0.0
                              );

   vec4 ContourScale = vec4 ( (ContourXX.x>0.0) ? ContourXX.x : 1.0,
                              (ContourXX.y>0.0) ? ContourXX.y : 1.0,
                              (ContourXX.z>0.0) ? ContourXX.z : 1.0,
                              (ContourXX.w>0.0) ? ContourXX.w : 1.0
                             );

// By default the full spectrum red, yellow, green, cyan, blue, magenta, red  is
// mapped to values of theta corresponding to the range 0 to 360  degrees, which
// is regarded as a unit turn. This can be changed by specifying a no zero value
// for the ColourScale parameter.  Normaly  this should be a small integer value
// in which case it gives the number of times the spectrum will be repeated  for
// a full 360 degree change in theta. Its default value is 1.0.

uniform float ColourScale;

   float ColourScaleFactor = ( ColourScale == 0.0 ) ? 1.0 : ColourScale;

/////////////////////////////////////////////////////////////////////////////////

// Control inputs. Part 3 - Parameters for the texture based colour schemes.

// The ColouringTexture input is only used if one of the texture based  colouring
// schemes is selected.  If required it is specified indirectly in the scene file
// as a source image for whatever node the shader is being used from,  i.e.  by a
// source: clause.

uniform sampler2D ColouringTexture; // Specified using a source: clause.

/////////////////////////////////////////////////////////////////////////////////

// Created by David Bargo - davidbargo/2015. Modified by TheEmu.
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

// -----------------------------------------------------------------------------

// A few useful constants.

float pi = radians(180.0);

const vec2 z_i   = vec2(0.0,1.0);
const vec2 z_ONE = vec2(1.0,0.0);
const vec2 z_TWO = vec2(2.0,0.0);
const vec2 z_SIX = vec2(6.0,0.0);

// -----------------------------------------------------------------------------

// Some basic functions. These are used in the set of functions to be plotted.

vec2 iz ( vec2 c ) { return vec2 ( -c.y, c.x ); }

vec2 mulz ( vec2 c, vec2 d ) { return c * mat2(d.x,-d.y,d.y,d.x); }
vec2 mulz ( vec2 c, vec2 d, vec2 e ) { return mulz ( c, mulz(d,e) ); }

vec2 divz ( vec2 c, vec2 d ) { return c * mat2(d.x,d.y,-d.y,d.x) / dot(d,d); }

vec2 invz ( vec2 c ) { return vec2 ( c.x,-c.y ) / dot(c,c); }

vec2 sinz ( vec2 c ) { return vec2 ( sin(c.x)*cosh(c.y),  cos(c.x)*sinh(c.y) ); }
vec2 cosz ( vec2 c ) { return vec2 ( cos(c.x)*cosh(c.y), -sin(c.x)*sinh(c.y) ); }
vec2 tanz ( vec2 c ) { return divz ( sinz(c), cosz(c) ); }
vec2 secz ( vec2 c ) { return invz ( cosz(c) ); }
vec2 cscz ( vec2 c ) { return invz ( sinz(c) ); }
vec2 cotz ( vec2 c ) { return invz ( tanz(c) ); }

vec2 sinhz ( vec2 c ) { return sinz ( -iz(c) ); } // sinh(z) = sin(i*z)/i
vec2 coshz ( vec2 c ) { return cosz ( +iz(c) ); } // cosh(z) = cos(i*z)
vec2 tanhz ( vec2 c ) { return divz ( sinhz(c), coshz(c) ); }
vec2 sechz ( vec2 c ) { return invz ( coshz(c) ); }
vec2 cschz ( vec2 c ) { return invz ( sinhz(c) ); }
vec2 cothz ( vec2 c ) { return invz ( tanhz(c) ); }

vec2 sqrtz ( vec2 c ) { float n = length(c)+c.x; return vec2(n,c.y)/sqrt(2.0*n); }

vec2 logz  ( vec2 c ) { return vec2 ( log(sqrt(dot(c,c))), atan(c.y,c.x) ); }
vec2 exp2z ( vec2 c ) { return vec2 ( c.x*c.x - c.y*c.y, 2.0*c.x*c.y ); }
vec2 epowz ( vec2 c ) { return vec2 ( cos(c.y), sin(c.y))*exp(c.x); }

vec2 sum_of_negative_powers ( vec2 c, int n )
 { vec2 t1 = invz(c);
   vec2 t2 = t1;
   vec2 result = t1;
   for ( int i=0; i<n; i++ )
    { t2 = mulz(t2,t1);
      result += t2;
    }
   return result;
 }

// -----------------------------------------------------------------------------

// The set of complex functions.  Each  of the supported functions is identified
// by a function id and defined by a macro named FUNC_nn where nn is the id. New
// functions may be defined by extending this list and adding it to  the  switch
// statement in func below. There is no particular reason for the particular set
// of functions chosen other than that they comprise the basic complex functions
// plus a few more complicated examples.  The identity function, FUNC_01 f(c)=c,
// has been included as it provides a useful reference point.
//
// Note, although the GLSL vector addition and subtraction operators + and - are
// equivalent to the complex addition and subtraction operators the same is  not
// true for the multiplication and division operators and it is necessary to use
// the mulz and divz functions in their place, or invz for 1.0/z. Similarly  the
// normal GLSL sqrt,exp, pow, sin, cos functions etc. are not applicable and the
// corresponding complex functions defined above have to be used instead.  As an
// alternative we could have defined a complex type and overloaded the functions
// and operators, but have not yet done so.

#define FUNC_01(c) (c) 
#define FUNC_02(c) (c*c)

#define FUNC_03(c) invz  ( c )
#define FUNC_04(c) invz  ( c + z_ONE ) + invz  ( c - z_ONE )
#define FUNC_05(c) invz  ( c + z_ONE ) - invz  ( c - z_ONE )
#define FUNC_06(c) invz  ( sqrtz(c) )

#define FUNC_07(c) sqrtz ( c )
#define FUNC_08(c) sqrtz ( z_ONE + c )
#define FUNC_09(c) sqrtz ( z_ONE + c ) + sqrtz ( z_ONE - c )
#define FUNC_10(c) sqrtz ( z_ONE + c ) - sqrtz ( z_ONE - c )

#define FUNC_11(c) epowz ( c )
#define FUNC_12(c) epowz ( invz(c) )
#define FUNC_13(c) exp2z ( c )
#define FUNC_14(c) exp2z ( invz(c) )
#define FUNC_15(c) invz  ( z_ONE + epowz(c.yx) )
#define FUNC_16(c) invz  ( z_ONE - epowz(c.yx) )

#define FUNC_17(c) logz ( c )
#define FUNC_18(c) logz ( invz(c) )
#define FUNC_19(c) logz ( c + invz(c) )
#define FUNC_20(c) logz ( c - invz(c) )
#define FUNC_21(c) logz ( z_ONE + invz(c) )
#define FUNC_22(c) logz ( z_ONE - invz(c) )

#define FUNC_23(c) sinz ( c )
#define FUNC_24(c) cosz ( c )
#define FUNC_25(c) tanz ( c )
#define FUNC_26(c) secz ( c )
#define FUNC_27(c) cscz ( c )
#define FUNC_28(c) cotz ( c )

#define FUNC_29(c) sinz ( sinz(c) )
#define FUNC_30(c) sinz ( cosz(c) )
#define FUNC_31(c) sinz ( tanz(c) )
#define FUNC_32(c) sinz ( secz(c) )
#define FUNC_33(c) sinz ( cscz(c) )
#define FUNC_34(c) sinz ( cotz(c) )

#define FUNC_35(c) mulz ( sinz(c), sinz(c) )
#define FUNC_36(c) mulz ( sinz(c), cosz(c) )
#define FUNC_37(c) mulz ( sinz(c), tanz(c) )
#define FUNC_38(c) mulz ( sinz(c), cotz(c) )
#define FUNC_39(c) mulz ( sinz(c), secz(c) )

#define FUNC_40(c) tanz ( sinz(c) )
#define FUNC_41(c) tanz ( cosz(c) )
#define FUNC_42(c) tanz ( tanz(c) )
#define FUNC_43(c) tanz ( secz(c) )
#define FUNC_44(c) tanz ( cscz(c) )
#define FUNC_45(c) tanz ( cotz(c) )

#define FUNC_46(c) sinhz ( c )
#define FUNC_47(c) coshz ( c )
#define FUNC_48(c) tanhz ( c )
#define FUNC_49(c) sechz ( c )
#define FUNC_50(c) cschz ( c )
#define FUNC_51(c) cothz ( c )

#define FUNC_52(c) divz  ( tanz(exp2z(c)), c )
#define FUNC_53(c) sinz  ( cosz(sinz(c)) )
#define FUNC_54(c) epowz ( invz(sqrtz(-c)) )
#define FUNC_55(c) epowz ( sinz(epowz(cosz(c))) )
#define FUNC_56(c) divz  ( sinz(c), c )
#define FUNC_57(c) divz  ( sinz(c), cosz(exp2z(c)) )
#define FUNC_58(c) divz  ( sqrtz ( c + z_ONE ), sqrtz ( c - z_ONE ) )
#define FUNC_59(c) invz  ( z_ONE + mulz(c,exp2z(exp2z(c))) )
#define FUNC_60(c) sqrtz ( divz ( logz(iz(c)-z_SIX), logz(iz(c)+z_TWO) ) )

// Note, in the following sin(c) is NOT the sin of the complex number c  but
// (sin(c.x),sin(c.y). Likewise the * operator is not complex multiplication
// but the normal component wise operator a*b = (a.x*b.x,a.y*b.y). These are
// not "proper" complex functions and do not necessarily result in conformal
// (angle preserving) mappings from their domains to co-domains

#define FUNC_61(c) ( sin(c) * sin(c) )
#define FUNC_62(c) ( sin(c) * cos(c) )
#define FUNC_63(c) ( sin(c) * tan(c) )
#define FUNC_64(c) ( sin(c) * sinz(c) )
#define FUNC_65(c) ( sin(c) * cosz(c) )
#define FUNC_66(c) ( sin(c) * tanz(c) )
#define FUNC_67(c) ( sin(c) * secz(c) )
#define FUNC_68(c) ( sin(c) * sinhz(c) )
#define FUNC_69(c) ( sin(c) * coshz(c) )
#define FUNC_70(c) ( sin(c) * tanhz(c) )
#define FUNC_71(c) ( sin(c) * sechz(c) )

#define FUNC_72(c) ( c * epowz(z_i*c.x) )
#define FUNC_73(c) ( c * epowz(z_i*c.y) )
#define FUNC_74(c) ( c * epowz(z_i*(c.x+c.y)) )

#define FUNC_75(c) ( exp(c.x)*cosz(c) )
#define FUNC_76(c) ( exp(c.x)*sinz(c) )

// Back to normal, proper, complex functions.

#define FUNC_77(c) ( mulz(c,c,c) + z_ONE )
#define FUNC_78(c) ( divz ( mulz(c-z_ONE,c+z_ONE,c+z_ONE), mulz(c+z_i,c-z_i,c-z_i) ) )
#define FUNC_79(c) ( mulz ( c/4.0, sinz(invz(c/4.0)) ) )
#define FUNC_80(c) ( mulz ( c, mulz(c,c), mulz(c,c) ) - z_ONE )
#define FUNC_81(c) ( invz ( mulz ( c, mulz(c,c), mulz(c,c) ) ) - z_ONE )
#define FUNC_82(c) ( ( c + invz(c) ) / 2.0 )
#define FUNC_83(c) ( mulz  ( sqrtz ( c + z_ONE ), sqrtz ( c - z_ONE ) ) )
#define FUNC_84(c) ( divz ( cosz(c), sinz(mulz(mulz(c,c),mulz(c,c))) ) )
#define FUNC_85(c) ( sum_of_negative_powers(c,5) )
#define FUNC_86(c) ( invz ( sum_of_negative_powers(c,5) ) )
#define FUNC_87(c) ( sinz(invz(c)) )
#define FUNC_88(c) ( (z_ONE+z_i)*sinz(c) )
#define FUNC_89(c) ( sinz(mulz(c,c,c)) - z_ONE )
#define FUNC_90(c) ( divz ( sinz(mulz(c,c,c)) - z_ONE, c ) )
#define FUNC_91(c) ( divz ( c - z_ONE, mulz(c,c) + c + z_ONE ) )
#define FUNC_92(c) ( divz ( cosz(c), sinz(mulz(mulz(c,c),mulz(c,c))-z_ONE ) ) )
#define FUNC_93(c) ( logz ( sinz(c) ) )
#define FUNC_94(c) ( logz ( sinhz(c) ) )
#define FUNC_95(c) ( logz ( tanhz(c) ) )
#define FUNC_96(c) ( (z_ONE+z_i) * logz(sinz(divz(mulz(c,c,c)-z_ONE,c))) )


// If you add more functions remember to update the LAST_FUNC_ID and to add
// the function evaluation to the list in the switch statement in func.

#define LAST_FUNC_ID 96 // REMEMBER TO UPDATE THIS

// -----------------------------------------------------------------------------

// funcId determines which of the functions is to be evaluated. This may be a
// single function (if FunctionDuration is 0.0) or may cycle through the list
// of functions changing every FunctionDuration seconds.

int funcId ( void )
 {
   int fn = FunctionId;

   if ( FunctionDuration != 0.0 )
    {
      float t = u_Elapsed/FunctionDuration;
      float f = mod ( float(FunctionId) - 1.0 + t, float(LAST_FUNC_ID) );
      fn = int(f) + 1;
    }

   return fn;

 }

// -----------------------------------------------------------------------------

// func evaluates the currently selected function. If you add a new function
// you need to add it here as well as declaring it above.

vec2 func ( vec2 c )
 {
   vec2 fz = vec2(0.0); // Default - used for an invalid Function Id

   // When using my laptop running VGHD with its Intel Integrated Graphics
   // processor then there is no problem with a single switch statement to
   // handle all the functions, currently 71 of them.  But, when using the
   // Nvidia GPU there is a limitation and only up to 64 functions can  be
   // handled by a single switch statement.  The  error reported being "IF
   // statement nested too deeply". To overcome this I use multiple switch
   // statements,  and  in case some other GPU driver has smaller limits I
   // handle the functions in groups of 20.

   int id = funcId(); 

   switch ( id )
    { case  1: fz = FUNC_01(c); break;
      case  2: fz = FUNC_02(c); break;
      case  3: fz = FUNC_03(c); break;
      case  4: fz = FUNC_04(c); break;
      case  5: fz = FUNC_05(c); break;
      case  6: fz = FUNC_06(c); break;
      case  7: fz = FUNC_07(c); break;
      case  8: fz = FUNC_08(c); break;
      case  9: fz = FUNC_09(c); break;
      case 10: fz = FUNC_10(c); break;
      case 11: fz = FUNC_11(c); break;
      case 12: fz = FUNC_12(c); break;
      case 13: fz = FUNC_13(c); break;
      case 14: fz = FUNC_14(c); break;
      case 15: fz = FUNC_15(c); break;
      case 16: fz = FUNC_16(c); break;
      case 17: fz = FUNC_17(c); break;
      case 18: fz = FUNC_18(c); break;
      case 19: fz = FUNC_19(c); break;
      case 20: fz = FUNC_20(c); break;
    }

   switch ( id )
    { case 21: fz = FUNC_21(c); break;
      case 22: fz = FUNC_22(c); break;
      case 23: fz = FUNC_23(c); break;
      case 24: fz = FUNC_24(c); break;
      case 25: fz = FUNC_25(c); break;
      case 26: fz = FUNC_26(c); break;
      case 27: fz = FUNC_27(c); break;
      case 28: fz = FUNC_28(c); break;
      case 29: fz = FUNC_29(c); break;
      case 30: fz = FUNC_30(c); break;
      case 31: fz = FUNC_31(c); break;
      case 32: fz = FUNC_32(c); break;
      case 33: fz = FUNC_33(c); break;
      case 34: fz = FUNC_34(c); break;
      case 35: fz = FUNC_35(c); break;
      case 36: fz = FUNC_36(c); break;
      case 37: fz = FUNC_37(c); break;
      case 38: fz = FUNC_38(c); break;
      case 39: fz = FUNC_39(c); break;
      case 40: fz = FUNC_40(c); break;
    }

   switch ( id )
    { case 41: fz = FUNC_41(c); break;
      case 42: fz = FUNC_42(c); break;
      case 43: fz = FUNC_43(c); break;
      case 44: fz = FUNC_44(c); break;
      case 45: fz = FUNC_45(c); break;
      case 46: fz = FUNC_46(c); break;
      case 47: fz = FUNC_47(c); break;
      case 48: fz = FUNC_48(c); break;
      case 49: fz = FUNC_49(c); break;
      case 50: fz = FUNC_50(c); break;
      case 51: fz = FUNC_51(c); break;
      case 52: fz = FUNC_52(c); break;
      case 53: fz = FUNC_53(c); break;
      case 54: fz = FUNC_54(c); break;
      case 55: fz = FUNC_55(c); break;
      case 56: fz = FUNC_56(c); break;
      case 57: fz = FUNC_57(c); break;
      case 58: fz = FUNC_58(c); break;
      case 59: fz = FUNC_59(c); break;
      case 60: fz = FUNC_60(c); break;
    }

   switch ( id )
    { case 61: fz = FUNC_61(c); break;
      case 62: fz = FUNC_62(c); break;
      case 63: fz = FUNC_63(c); break;
      case 64: fz = FUNC_64(c); break;
      case 65: fz = FUNC_65(c); break;
      case 66: fz = FUNC_66(c); break;
      case 67: fz = FUNC_67(c); break;
      case 68: fz = FUNC_68(c); break;
      case 69: fz = FUNC_69(c); break;
      case 70: fz = FUNC_70(c); break;
      case 71: fz = FUNC_71(c); break;
      case 72: fz = FUNC_72(c); break;
      case 73: fz = FUNC_73(c); break;
      case 74: fz = FUNC_74(c); break;
      case 75: fz = FUNC_75(c); break;
      case 76: fz = FUNC_76(c); break;
      case 77: fz = FUNC_77(c); break;
      case 78: fz = FUNC_78(c); break;
      case 79: fz = FUNC_79(c); break;
      case 80: fz = FUNC_80(c); break;
    }

   switch ( id )
    { case 81: fz = FUNC_81(c); break;
      case 82: fz = FUNC_82(c); break;
      case 83: fz = FUNC_83(c); break;
      case 84: fz = FUNC_84(c); break;
      case 85: fz = FUNC_85(c); break;
      case 86: fz = FUNC_86(c); break;
      case 87: fz = FUNC_87(c); break;
      case 88: fz = FUNC_88(c); break;
      case 89: fz = FUNC_89(c); break;
      case 90: fz = FUNC_90(c); break;
      case 91: fz = FUNC_91(c); break;
      case 92: fz = FUNC_92(c); break;
      case 93: fz = FUNC_93(c); break;
      case 94: fz = FUNC_94(c); break;
      case 95: fz = FUNC_95(c); break;
      case 96: fz = FUNC_96(c); break;
     }

    return fz;

 }

// -----------------------------------------------------------------------------

// Convert from cartesian to polar representation.

vec2 polar ( vec2 xy )
 {
   float r = length(xy);
   float a = atan(xy.y,xy.x) / (2.0*pi);
   return vec2 ( r, a );
 }

// -----------------------------------------------------------------------------

// Convert a colour from a (hue,saturation,value) representation to its  red,
// green, blue representation.  Based  on iq's smooth hsv to rgb but with hue
// being rescaled so that it is expressed in terms of whole turns rather than
// radians.

vec3 hsv2rgb( in vec3 c )
 {
   vec3 k = vec3(0.0,4.0,2.0);
   vec3 rgb = abs ( mod(c.x*6.0+k,6.0) - 3.0 ) - 1.0;
   rgb = clamp ( rgb, 0.0, 1.0 );
   rgb = rgb*rgb * ( 3.0 - 2.0*rgb );
   return c.z * mix ( vec3(1.0), rgb, c.y );
 }

// -----------------------------------------------------------------------------

// The default colouring is based on the value that has been evaluated for the
// currently selected complex function. It uses two colouring schemes at once.

// The  first is based on the r * exp(i*theta) representation of  the  complex
// value z and, by default, produces a rainbow as a function of theta with the
// Red, Yellow, Green, Cyan, Blue and Magenta segments being centered at theta
// corresponding to 0, 60, 120, 180, 240 and 300 degrees repectively. A set of
// whilte lines is suuperimposed which are,  by default,  centered  on integer
// values of r. The default spacing of the colour bands and white lines can be
// modified as described in the comments at the start of this source file.
//
// The second colouring used is based on the cartesian representation  x + i*y
// of z and, by default, produces dark lines at integer values of x and y. The
// default spacing of the dark lines can be modified as described by  comments
// at the start of this source file.
//
// The sets of dark and white lines may be regarded as x, y and r contours. By
// default no theta contours are drawn as they can tend to wash out the colour
// bands, however they can be enabled via the contour control parameters as is
// described in the comments at the start of this source file.
//
// Technicaly the colouring is performed by working in the HSV (hue,saturation
// and value) colour system with theta selection the basic hue which then  has
// its saturation (darkness) modulated by the distance from (x,y) contours and
// its value (whiteness) component is modulated by the distance from (r,theta)
// contours. By default the modulation produces rather fuzzy contours, this is
// deliberately so for artistic reasons though their widths do indicate to the
// rate of change of the function with broad areas indicating flat regions  of
// the co-domain. The sharpness of the contours may be altered as is described
// in the comments at the start of this source file.
//
// The arguments z and p should normaly be the cartesian and equivalent  polar
// representations of the same complex number,  but for an alternate colouring
// scheme this rule can be broken.  If it is broken then the interpretation of
// the relationship between the two sets of contours is more complicated  than
// when the nominal relationship between the arguments holds.
//
// This function is based on Huw Bowles version but with minor enhancements.

vec4 simpleColouring ( vec2 z, vec2 p )
 {
   // Determine the colour based on the theta part of the
   // polar representation of z where z = r * i*theta.

   float hue = p.y * ColourScaleFactor; // N.B. the angle is in "turns".

   // Determine modulation factors for x and y contours.

   vec2 xy = z * ContourScale.xy;
   xy = abs(fract(xy)-0.5) - 0.25;
   xy = step(0.0,xy) * xy * 4.0;
   xy = xy * ContourEnable.xy;
   xy = 1.0 - pow(abs(xy),vec2(ContourSharp.x));

   // Determine modulation factors for r and theta contours.

   vec2 ra = p * ContourScale.zw;
   ra = abs(fract(ra)-0.5) - 0.25;
   ra = step(0.0,ra) * ra * 4.0;
   ra = ra * ContourEnable.zw;
   ra = 1.0 - pow(abs(ra),vec2(ContourSharp.y));

   // Apply the contour modulation factors.

   float sat = ra.x * ra.y;
   float val = xy.x * xy.y;

   val = mix ( 1.0, val, sat*0.5 );

   // Convert from HSV colour representation to RGB.

   return vec4 ( hsv2rgb(vec3(hue,sat,val)), 1.0 );

 }

// -----------------------------------------------------------------------------

// As an alternative to the default simple colouring scheme the image may be
// coloured by using the function's components to index into a texture. This
// will result in the final image comprising a tiling of distorted copies of
// the texture image. As a simple tiling would leave discontinuities at  the
// edges of each tile (unless the tiles were designed to avoid it) alternate
// tiles are optionaly flipped so that there is continuity of colour  across
// tile boundaries thereby improving the look of the final  image.  Flipping
// is optional because some textures may be deliberately  directional,  e.g.
// have an arrow on them, and it is desired to maintain this directionality.

vec4 texturedColours ( vec2 z, bool flip )
 {
   vec2 zz;

   if ( flip )
    {
      // Reduce z.x and z.y to each be in range 0.0 to 2.0

      zz = mod ( z, 2.0 );

      zz.x += (zz.x<0.0) ? 2.0 : 0.0;
      zz.y += (zz.y<0.0) ? 2.0 : 0.0;

      // Reduce z.x and z.y to be in range 0.0 to 1.0  by
      // folding about the x=1.0 and y=1.0 axes.  This is
      // done to ensure colour continuity across each  of
      // the unit squares that are  used  to  "tile"  the
      // the image thereby reducing the visibility of the
      // individual tile boundaries.

      zz.x = (zz.x<1.0) ? zz.x : 2.0 - zz.x;
      zz.y = (zz.y<1.0) ? zz.y : 2.0 - zz.y;
    }
   else
    {
      zz = fract(z);
    }

   // Lookup the colour to use for the current pixel.

   return texture2D ( ColouringTexture, zz );

 }

// -----------------------------------------------------------------------------

// The main routine converts the pixel position to a complex  number,  z,  with
// z = (0.0,0.0) being at the center of the window. The scaling is such that if
// the window is wider than it is high z = (0,pi) and z = (0,-pi)  will  be  at
// the middle of the top and bottom of the window respectively with  z = (pi,0)
// and z = (-pi,0) both lying within the window.  However, if the window height
// is larger than its width this is modified so that z = (pi,0) and z = (-pi,0)
// still lie within the window and a greater range of the y component o f z  is
// covered. This is done because the (pi,0) and (-pi,0) points are rather  more
// interesting than the (0,pi) and (0,-pi) points for the  functions  that  are
// considered here.
//
// After determining z it is used as the argument  to  the  currently  selected
// function, f, yeilding a second complex number fz = f(z).  This  is subjected
// to a RotAlpha such that if fz = r * exp(i*theta) theta goes  to  theta+Alpha
// where Alpha is a linear function of time.
//
// Finaly the function value, fz, is used as the basis for colouring the  final
// image.  This may be based on a texture lookup or may use the default  colour
// scheme.
//
// Note, in general complex functions of complex variable are multi-valued.  In
// this program only the principal values are considered.  Where  this involves
// cuts in the function value these can often be seen as discontinuities in the
// colouring across the cuts, e.g. a jump from red to green. However, depending
// on the function and the use of the ColourScale parameter the colours on both
// sides of a cut can "accidently" match disguising the cut or there may  be  a
// colour discontinuity where there is no cut,  e.g. if ColourScale is set to a
// none integer value.

void main ( void )
 {
   // Get the function's argument.

   float s = min ( u_WindowSize.x*1.3, u_WindowSize.y );
   vec2  z = ( 2.0*gl_FragCoord.xy - u_WindowSize ) * pi / s;

   // Apply the domain modifiers.

   z = (z+PreAdd) * DomainScale;

   // Evaluate the function and apply time dependant RotAlpha.

   vec2 fz = func(z) * RotAlpha;

   // Apply the codomain modifiers.

   fz = (fz+PostAdd) * CodomainScale;

   // Get the equivalent polar form of the function's value.

   vec2 pz = polar(fz);

   // Colour the current pixel according to the function's value
   // and the selected colouring scheme. Note, in some cases the
   // roles of fz and pz are swapped and the normal relationship
   // expected between the two parameters of simpleColouring has
   // been deliberately broken.

   switch ( ColourScheme )
    { // Simple internal colour schemes.
      case  0: gl_FragColor = simpleColouring ( fz, pz );        break;
      case  1: gl_FragColor = simpleColouring ( pz, polar(pz) ); break;
      case  2: gl_FragColor = simpleColouring ( pz, fz );        break;
      // Texture based colour schemes.
      case 10: gl_FragColor = texturedColours ( fz, true  );     break;
      case 11: gl_FragColor = texturedColours ( fz, false );     break;
      case 12: gl_FragColor = texturedColours ( pz, true  );     break;
      case 13: gl_FragColor = texturedColours ( pz, false );     break;
    }
 }

// -----------------------------------------------------------------------------
